/*-------------------------------------------------------*
*  Name:      FCE filter for ZModeler v1.05              *
*  Purpose:   NFS4 files import/export                   *
*  Authors:   Oleg M.                                    *
*                                                        *
*  History:   16.Sep.2001 - rebuild for ZModeler v1.05   *
*                                                        *
*-------------------------------------------------------*/
#include <Struct.h>
#include <ZMuserAddons/z3dUserUnits.h>
#include <3DEngine.h>

HINSTANCE      dll, app;

#include "resource.h"
#include "change.h"
#include "floatbar.h"
#include "FCE.h"


//////////////////////////////////////////////////////////////
//  DWORD Capabilities - a combination versions
DWORD CALLBACK Capabilities(long)
{
  return ZMODELER_BUILD_VERSION();
}

//////////////////////////////////////////////////////////////
//  DWORD Supports returns a combination of supported features
DWORD CALLBACK Supports(DWORD)
{
  return  Z3D_PLUGSUPPORTS_IMPORT |
          Z3D_PLUGSUPPORTS_EXPORT |
          Z3D_PLUGSUPPORTS_PROCESSING |
          Z3D_PLUGSUPPORTS_MISCFLAGS;
}

//////////////////////////////////////////////////////////////
//  DWORD DynamicLoading specifies whether the dll will be
//  loaded only when it'sfunctions are needed.
//  You should NOT make it dynamic, if it is a processor-type.
DWORD CALLBACK DynamicLoading(DWORD)
{
  return 0;//non-dynamic because of floatbar!
}

//////////////////////////////////////////////////////////////
//  void GetFilterExtention returns a file extention for sup-
//  ported files. Used if it is an IMPORT/EXPORT type
char* CALLBACK GetFilterExtension(DWORD)
{
  return "fce";
}

//////////////////////////////////////////////////////////////
//  void GetFilterMask returns a file-mask for CFileDilaog.
//  Used if it is an IMPORT/EXPORT type.
char* CALLBACK GetFilterMask(DWORD)
{
  return "NFS4 FCE files (*.fce)|*.fce|";
}


//////////////////////////////////////////////////////////////
//  DWORD PlaceType returns a DWORD, that indicates, where the
//  plugin gode will be placed. It can be MenuBAR, MainMenu
DWORD CALLBACK PlaceType(DWORD)
{
  return Z3D_PROCPLACE_FLOATBAR;
}

//////////////////////////////////////////////////////////////
//  DWORD GetProcsAmount returns a number of IDENTICAL procedu-
//  res that will be placed in the same folder. I think there is
//  No limitation here... (256 is enough;)
DWORD CALLBACK GetProcsAmount(DWORD)
{
  return 1; //SINGLE FLOATBAR (FCE)
}

//////////////////////////////////////////////////////////////
// returns proc name how it's named internally and what it really
// stands for.
char* CALLBACK GetProgrammingProcName(DWORD num)
{
  if (num==0) return "FCE Custom Settings";
  return NULL;
}

//////////////////////////////////////////////////////////////
// returns proc name that is adapted for simple mode.
char* CALLBACK GetProcName(DWORD num)
{
  return GetProgrammingProcName(num);
}


char* CALLBACK GetProcComment(DWORD num)
{
  return "FCE custom settings\n";
}


DWORD CALLBACK PluginSafeLoad(tProcParams* params)
{
  float coef = 1.06f;
  params->sysp(
      (sRequestType)Z3D_USERREQUEST_ADDUNIT,
      (tObjectSet*)"meters",
      "NFS4 meters",
      (long*)&coef,
      NULL);
  return 0;
}

char* CALLBACK MiscellaneousFlagsMeaning(DWORD num)
{
  if (num!= 0) return NULL;
  return
    "Need for speed 4 FCE filter\n"
    "Rotateable vertex (:OD part only)\n\n\n\n"
    "\n\n\n\n"
    "\n\n\n\n";
}


/*************************************************************/
/*************************************************************/
/*************************************************************/
//////////////////////////////////////////////////////////////
//  INTERNAL DECLARATIONS GOES HERE.
#define NFS4FCEHEADERSIZE  8248
#define NFS4HEADERID1    0x00101014
#define MCO_HEADERID1    0x00101015
#define NFS4HEADERID2    0x00000000

typedef struct tFCEVector
{
  float X;
  float Y;
  float Z;
}tFCEVector;


typedef struct tFCETrian
{
  long  TexturePage;
  long  I1;
  long  I2;
  long  I3;
  char  Unknow[12];
  long  Smoothing;
  float  U1;
  float  U2;
  float  U3;
  float  V1;
  float  V2;
  float  V3;
}tFCETrian;


typedef struct tFCEColor
{
  BYTE Hue;
  BYTE Saturation;
  BYTE Brightness;
  BYTE Transparency;
}tFCEColor;


typedef struct nfs4fcehead
{

  long  HeaderID;
  long  HeaderID2;
  long  NumofTria;
  long  NumofVert;
  long  NumofArt;
  long  VertOffset;
  long  NormalsOffset;
  long  TriaOffset;
  long  TempStorage1offset;
  long  TempStorage2offset;
  long  TempStorage3offset;
  long  UndamagedVertOffset;
  long  UndamagedNormOffset;
  long  DamagedVertOffset;
  long  DamagedNormOffset;
  long  UnknowAreaOffset;
  long  MovementsOffset;
  long  Unknow1Offset;
  long  Unknow2Offset;
  float  RealXHalfSize;
  float  RealYHalfSize;
  float  RealZHalfSize;
  long  NumofDummies;
  tFCEVector  Dummies[16];
  long  NumParts;
  tFCEVector  Parts[64];
  long  Vert1stPart[64];
  long  VertNumPart[64];
  long  Tria1stPart[64];
  long  TriaNumPart[64];
  long  NumColors;
  tFCEColor  PrimColors[16];//tcolor 4 bytes
  tFCEColor  IntrColors[16];
  tFCEColor  SecnColors[16];
  tFCEColor  HairColors[16];
  char  UnknowSpace[260];
  char  DummyNames[16][64];//[16*64];
  char  PartNames[64][64];//[64*64];
  char  UnknowSector[528];

}nfs4fcehead;


//////////////////////////////////////////////////////////////
//  internal function: Creates a "Diamond" objects that speci-
//  fies the location of Dummie object - lights.

tObject* _stdcall CreatePiramid(float atx, float aty, float atz, char* name,  BOOL hidden)
{
  float edge = 0.05f;
  tObject *resObj;
  resObj = new tObject(name, 6, 8);

  resObj->VertTable->Table[0].MakeVertex(
    tPOINT(atx, aty+edge, atz),
    tPOINT(0.0f, 1.0f, 0.0f));
  resObj->VertTable->Table[1].MakeVertex(
    tPOINT(atx, aty, atz+edge),
    tPOINT(0.0f, 0.0f, 1.0f));
  resObj->VertTable->Table[2].MakeVertex(
    tPOINT(atx+edge, aty, atz),
    tPOINT(1.0f, 0.0f, 0.0f));
  resObj->VertTable->Table[3].MakeVertex(
    tPOINT(atx, aty, atz-edge),
    tPOINT(0.0f, 0.0f, -1.0f));
  resObj->VertTable->Table[4].MakeVertex(
    tPOINT(atx-edge, aty, atz),
    tPOINT(-1.0f, 0.0f, 0.0f));
  resObj->VertTable->Table[5].MakeVertex(
    tPOINT(atx, aty-edge, atz),
    tPOINT(0.0f, -1.0f, 0.0f));
  resObj->FaceTable->Table[0].MakeFace(0, 2, 1);
  resObj->FaceTable->Table[1].MakeFace(0, 3, 2);
  resObj->FaceTable->Table[2].MakeFace(0, 4, 3);
  resObj->FaceTable->Table[3].MakeFace(0, 1, 4);
  resObj->FaceTable->Table[4].MakeFace(5, 1, 2);
  resObj->FaceTable->Table[5].MakeFace(5, 2, 3);
  resObj->FaceTable->Table[6].MakeFace(5, 3, 4);
  resObj->FaceTable->Table[7].MakeFace(5, 4, 1);
  if (hidden) resObj->Hide();

  tPOINT center = tPOINT(atx, aty, atz);
  SetMatrixPosition(resObj->mTransform, center);
  return resObj;
}


//////////////////////////////////////////////////////////////
//  Shows a failure message.
void _stdcall ShowFailMessage(CWnd* pwnd, char *errmsg, char *cause, UINT icon)
{
  char str[512];
  strcpy(str, "Error importing/exporting file: ");
  strcat(str, errmsg);
  strcat(str, "\nPossible caused by: ");
  strcat(str, cause);
  pwnd->MessageBox(str,"Filter message:", icon);
}


/**********************************************************************/
//      NFS4 File Importer to Zanoza 3D Editor
/**********************************************************************/
DWORD CALLBACK Import(  CString fromfile,
              CWnd *pwnd,
            tObjectSet* Objects,
            tUnDataSet* UnData,
            CDirect3D*  d3d,
            SYSTEMREQUESTPROC RequestProc,
            HINSTANCE AppHIns,
            HINSTANCE DllHIns)
{
  CFile InFile;
  int i;

  BYTE      *TextureMap;
  CString      TextureFileName = fromfile.Left(fromfile.Find('.'));//00
  CString      CurrentFileName, MatNameStr;
  BOOL      LoadDamaged;

  if (!pwnd) pwnd = AfxGetMainWnd();
  LoadDamaged = (pwnd->MessageBox(
    "Load Damaged Parts?",
    "Loading...",
    MB_YESNO | MB_ICONQUESTION)==IDYES);

  if (!InFile.Open(fromfile, CFile::modeRead))
  {
    ShowFailMessage(pwnd, "Can not open the file", "Invalid file attributes", MB_ICONHAND);
    return 0;
  }


  /**************************************
  ******    Lets Read IT!!!    *******
  **************************************/
  nfs4fcehead FileHead;


  InFile.Read(&FileHead, sizeof(nfs4fcehead));
  
  if ((FileHead.HeaderID != NFS4HEADERID1) && (FileHead.HeaderID != MCO_HEADERID1))
  {
    ShowFailMessage(pwnd, "Invalid file format", "Not a NFS4 or MCO file format", MB_ICONHAND);
    return 0;
  }

  TextureMap = new BYTE[256];
  for (i=0; i < 256; i++) TextureMap[i] = 0xff;

  tFCEVector  *Vectors;
  tFCEVector  *Normals;
  tFCEVector  *DamVectors;
  tFCEVector  *DamNormals;
  tFCETrian  *Trian;
  Vectors = new tFCEVector[FileHead.NumofVert];
  Normals = new tFCEVector[FileHead.NumofVert];
  DamVectors = new tFCEVector[FileHead.NumofVert];
  DamNormals = new tFCEVector[FileHead.NumofVert];
  Trian  = new tFCETrian[FileHead.NumofTria];
  DWORD  *RotMap = new DWORD[FileHead.NumofVert];

  BYTE* pBuf = new BYTE[16*16+4];
  pBuf[0] = (BYTE)FileHead.NumColors;

  for (long count=0; count < 16; count++)
  {
    pBuf[1+count*16]   = FileHead.PrimColors[count].Hue;
    pBuf[1+count*16+1] = FileHead.PrimColors[count].Saturation;
    pBuf[1+count*16+2] = FileHead.PrimColors[count].Brightness;
    pBuf[1+count*16+3] = FileHead.PrimColors[count].Transparency;
    pBuf[1+count*16+4] = FileHead.IntrColors[count].Hue;
    pBuf[1+count*16+5] = FileHead.IntrColors[count].Saturation;
    pBuf[1+count*16+6] = FileHead.IntrColors[count].Brightness;
    pBuf[1+count*16+7] = FileHead.IntrColors[count].Transparency;
    pBuf[1+count*16+8] = FileHead.SecnColors[count].Hue;
    pBuf[1+count*16+9] = FileHead.SecnColors[count].Saturation;
    pBuf[1+count*16+10]= FileHead.SecnColors[count].Brightness;
    pBuf[1+count*16+11]= FileHead.SecnColors[count].Transparency;
    pBuf[1+count*16+12]= FileHead.HairColors[count].Hue;
    pBuf[1+count*16+13]= FileHead.HairColors[count].Saturation;
    pBuf[1+count*16+14]= FileHead.HairColors[count].Brightness;
    pBuf[1+count*16+15]= FileHead.HairColors[count].Transparency;
  }
  ///////////////////////////////////
  UnData->CreateRec(CString(CString("Import: ")+fromfile).GetBuffer(64),
        Z3D_UNDATA_COLORS,
        260,
        (char*)pBuf,
        260);
  ///////////////////////////////////

  /******* reading normal vertices  *********/
  InFile.Seek( NFS4FCEHEADERSIZE + FileHead.VertOffset, CFile::begin);
  InFile.ReadHuge(Vectors, sizeof(tFCEVector)*FileHead.NumofVert);
  /******* reading damaged vertices *********/
  InFile.Seek( NFS4FCEHEADERSIZE + FileHead.DamagedVertOffset, CFile::begin);
  InFile.ReadHuge(DamVectors, sizeof(tFCEVector)*FileHead.NumofVert);
  /******* reading normal normals   *********/  
  InFile.Seek( NFS4FCEHEADERSIZE + FileHead.NormalsOffset, CFile::begin);
  InFile.ReadHuge(Normals, sizeof(tFCEVector)*FileHead.NumofVert);
  /******* reading damaged normals  *********/  
  InFile.Seek( NFS4FCEHEADERSIZE + FileHead.DamagedNormOffset, CFile::begin);
  InFile.ReadHuge(DamNormals, sizeof(tFCEVector)*FileHead.NumofVert);
  /******* reading normal triangles *********/
  InFile.Seek( NFS4FCEHEADERSIZE + FileHead.TriaOffset, CFile::begin);
  InFile.ReadHuge(Trian, sizeof(tFCETrian)*FileHead.NumofTria);
  /********   reading Rotation Map   ********/
  InFile.Seek( NFS4FCEHEADERSIZE + FileHead.MovementsOffset, CFile::begin);
  InFile.ReadHuge(RotMap, sizeof(DWORD)*FileHead.NumofVert);

  long CountPart, CountV, CountT;

  for (CountPart = 0; CountPart < FileHead.NumofDummies; CountPart++)
  {//runing through dummies
    CString dName = "Dummy:";
    dName += FileHead.DummyNames[CountPart];
    tObject* pDummy = 
      CreatePiramid(
        -FileHead.Dummies[CountPart].X,
        FileHead.Dummies[CountPart].Y,
        FileHead.Dummies[CountPart].Z,
        dName.GetBuffer(64),//tmpname,
        /*hidden*/TRUE);

    Objects->AddObject(pDummy);
    delete pDummy;
  }//runing through dummies

  for (CountPart = 0; CountPart < FileHead.NumParts; CountPart++)
  {//runing through objects
    tObject* newObj;
    tObject* newObjdam;
    newObj = new tObject();
    newObj->VertTable->Init(FileHead.VertNumPart[CountPart]);
    newObj->FaceTable->Init(FileHead.TriaNumPart[CountPart]);
    if (LoadDamaged)
    {
      newObjdam = new tObject();
      newObjdam->Hide();
      newObjdam->VertTable->Init(FileHead.VertNumPart[CountPart]);
      newObjdam->FaceTable->Init(FileHead.TriaNumPart[CountPart]);
    }
    tPOINT center = tPOINT(-FileHead.Parts[CountPart].X,
                            FileHead.Parts[CountPart].Y,
                            FileHead.Parts[CountPart].Z);
    SetMatrixPosition(newObj->mTransform, center);
    if (LoadDamaged)
      SetMatrixPosition(newObjdam->mTransform, center);
    for (CountV = 0; CountV < FileHead.VertNumPart[CountPart]; CountV++)
    {//runing through vertices
      newObj->VertTable->Table[CountV].X = -(Vectors[FileHead.Vert1stPart[CountPart]+CountV].X+ FileHead.Parts[CountPart].X);
      newObj->VertTable->Table[CountV].Y = Vectors[FileHead.Vert1stPart[CountPart]+CountV].Y  + FileHead.Parts[CountPart].Y;
      newObj->VertTable->Table[CountV].Z = Vectors[FileHead.Vert1stPart[CountPart]+CountV].Z  + FileHead.Parts[CountPart].Z;
      if (RotMap[FileHead.Vert1stPart[CountPart]+CountV]==0x00000000) newObj->VertTable->Table[CountV].SetFlag(Z3D_FLAG_MISC1);
      newObj->VertTable->Table[CountV].NormalX =
          -Normals[FileHead.Vert1stPart[CountPart]+CountV].X;
      newObj->VertTable->Table[CountV].NormalY =
          Normals[FileHead.Vert1stPart[CountPart]+CountV].Y;
      newObj->VertTable->Table[CountV].NormalZ =
          Normals[FileHead.Vert1stPart[CountPart]+CountV].Z;
      /****************************/
      if (LoadDamaged)
      {
        newObjdam->VertTable->Table[CountV].X = -(DamVectors[FileHead.Vert1stPart[CountPart]+CountV].X  + FileHead.Parts[CountPart].X);
        newObjdam->VertTable->Table[CountV].Y = DamVectors[FileHead.Vert1stPart[CountPart]+CountV].Y  + FileHead.Parts[CountPart].Y;
        newObjdam->VertTable->Table[CountV].Z = DamVectors[FileHead.Vert1stPart[CountPart]+CountV].Z  + FileHead.Parts[CountPart].Z;
        newObjdam->VertTable->Table[CountV].NormalX =
            -DamNormals[FileHead.Vert1stPart[CountPart]+CountV].X;
        newObjdam->VertTable->Table[CountV].NormalY =
            DamNormals[FileHead.Vert1stPart[CountPart]+CountV].Y;
        newObjdam->VertTable->Table[CountV].NormalZ =
            DamNormals[FileHead.Vert1stPart[CountPart]+CountV].Z;
      }//damaged
    }
    for (CountT = 0; CountT < FileHead.TriaNumPart[CountPart]; CountT++)
    {//runing through vertices
      if (((BYTE)Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage < 256) &&
        (TextureMap[(BYTE)Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage]==0xff))
      {//this texture never apperes before;
        if ((BYTE)Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage < 10)
          CurrentFileName = TextureFileName + "0"+Int2String(Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage)+".tga";
        else
          CurrentFileName = TextureFileName+Int2String(Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage)+".tga";
        if (CurrentFileName.GetLength() > 0)
        {
          if (CurrentFileName.ReverseFind('\\') > 0)
            MatNameStr = "Textured Material: "+CurrentFileName.Right(CurrentFileName.GetLength()-CurrentFileName.ReverseFind('\\'));
          else
            MatNameStr = "Textured Material: "+CurrentFileName;
          TextureMap[(BYTE)Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage] =
            (BYTE)d3d->CreateMaterial(
              1.0f,
              1.0f,
              1.0f,
              1.0f,
              1.0f,
              80.0f,
              (char*)CurrentFileName.GetBuffer(512),
              "",
              "chrome.bmp",
              "",
              (char*)MatNameStr.GetBuffer(MAX_MATERIALNAMELEN),
              0,
              D3DTOP_MODULATE ,
              D3DTOP_DISABLE,
              D3DTOP_ADD,
              D3DTOP_DISABLE,
              D3DBLEND_SRCALPHA,
              D3DBLEND_INVSRCALPHA,
              2,
              0x20,
              D3DCMP_GREATEREQUAL);
        }
        else
          TextureMap[(BYTE)Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage] = 0;
      }
      newObj->FaceTable->Table[CountT].I1 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].I1;
      newObj->FaceTable->Table[CountT].I2 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].I3;
      newObj->FaceTable->Table[CountT].I3 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].I2;
      if (Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage > 255)
        newObj->FaceTable->Table[CountT].Material = 0;
      else
        newObj->FaceTable->Table[CountT].Material = TextureMap[(BYTE)Trian[FileHead.Tria1stPart[CountPart]+CountT].TexturePage];
      newObj->FaceTable->Table[CountT].nRenderFlags = 0;
      newObj->FaceTable->Table[CountT].SetMiscFlag(0, Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & ~0x0100000F);
      if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x01) == 0x01)//nochrome
        newObj->FaceTable->Table[CountT].nRenderFlags |= NoBlend;
      if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x02) == 0x02)//high chrome
        newObj->FaceTable->Table[CountT].nRenderFlags |= HighBlend;
      if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x04) == 0x04)//nocull
        newObj->FaceTable->Table[CountT].nRenderFlags |= CullNone;
      if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x08) == 0x08)//semitrans
        newObj->FaceTable->Table[CountT].nRenderFlags |= SemiTransparent;

      newObj->FaceTable->Table[CountT].U1 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].U1;
      newObj->FaceTable->Table[CountT].U2 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].U3;
      newObj->FaceTable->Table[CountT].U3 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].U2;
      newObj->FaceTable->Table[CountT].V1 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].V1;
      newObj->FaceTable->Table[CountT].V2 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].V3;
      newObj->FaceTable->Table[CountT].V3 =
          Trian[FileHead.Tria1stPart[CountPart]+CountT].V2;
      /****************** damaged  ********************/
      if (LoadDamaged)
      {
        newObjdam->FaceTable->Table[CountT].I1 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].I1;
        newObjdam->FaceTable->Table[CountT].I2 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].I3;
        newObjdam->FaceTable->Table[CountT].I3 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].I2;
        newObjdam->FaceTable->Table[CountT].Material =
          newObj->FaceTable->Table[CountT].Material;
        newObjdam->FaceTable->Table[CountT].SetMiscFlag(0, Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & ~0x0100000F);
        newObjdam->FaceTable->Table[CountT].nRenderFlags = 0;

        if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x01) == 0x01)//nochrome
          newObjdam->FaceTable->Table[CountT].nRenderFlags |= NoBlend;
        if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x02) == 0x02)//HighChrome
          newObjdam->FaceTable->Table[CountT].nRenderFlags |= HighBlend;
        if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x04) == 0x04)//nocull
          newObjdam->FaceTable->Table[CountT].nRenderFlags |= CullNone;
        if ((Trian[FileHead.Tria1stPart[CountPart]+CountT].Smoothing & 0x08) == 0x08)//semitrans
          newObjdam->FaceTable->Table[CountT].nRenderFlags |= SemiTransparent;

        newObjdam->FaceTable->Table[CountT].U1 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].U1;
        newObjdam->FaceTable->Table[CountT].U2 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].U3;
        newObjdam->FaceTable->Table[CountT].U3 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].U2;
        newObjdam->FaceTable->Table[CountT].V1 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].V1;
        newObjdam->FaceTable->Table[CountT].V2 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].V3;
        newObjdam->FaceTable->Table[CountT].V3 =
            Trian[FileHead.Tria1stPart[CountPart]+CountT].V2;
      }//damaged
    }
    CString ObjName = FileHead.PartNames[CountPart];
    CString DamObjName = ObjName+CString("[D]");
    strcpy(newObj->ObjectName, ObjName.GetBuffer(MAX_OBJECT_NAMELENGTH));
    Objects->AddObject(newObj);
    delete newObj;
    if (LoadDamaged) 
    {
      strcpy(newObjdam->ObjectName, DamObjName.GetBuffer(MAX_OBJECT_NAMELENGTH));
      Objects->AddObject(newObjdam);
      delete newObjdam;
    }
  }//runing through objects
  if (Vectors)  delete[] Vectors;
  if (Normals)  delete[] Normals;
  if (DamVectors)  delete[] DamVectors;
  if (DamNormals)  delete[] DamNormals;
  if (Trian)    delete[] Trian;
  if (RotMap)    delete[] RotMap;
  if (TextureMap)  delete[] TextureMap;

  InFile.Close();
  return 1;
}//Import



/**********************************************************************/
/**********************************************************************/
//      NFS4 File Exporter to Zanoza 3D Editor
/**********************************************************************/
/**********************************************************************/
extern DWORD CALLBACK Export(  CString Tofile,
                  CWnd* pwnd,
                  tObjectSet* Objects,
                  tUnDataSet* UnData,
                  CDirect3D*  d3d,
                  SYSTEMREQUESTPROC RequestProc,
                  HINSTANCE AppHIns,
                  HINSTANCE DllHIns)
{
  CFile OutFile;
  if (!OutFile.Open(Tofile, CFile::modeWrite | CFile::modeCreate))
  {
    ShowFailMessage(pwnd, "Error rewriting file","Invalid Attributes?",MB_ICONQUESTION);
    return 0;
  }
/*
  long  HeaderID;      +
  long  HeaderID2;      +
  long  NumofTria;      ++
  long  NumofVert;      ++
  long  NumofArt;      +
  long  VertOffset;      +
  long  NormalsOffset;    +
  long  TriaOffset;      +
  long  TempStorage1offset  +
  long  TempStorage2offset;  +
  long  TempStorage3offset;  +
  long  UndamagedVertOffset;+
  long  UndamagedNormOffset;+
  long  DamagedVertOffset;  +
  long  DamagedNormOffset;  +
  long  UnknowAreaOffset;  +
  long  MovementsOffset;  ++
  long  Unknow1Offset;    +
  long  Unknow2Offset;    +
  float  RealXHalfSize;    ++
  float  RealYHalfSize;    ++
  float  RealZHalfSize;    ++
  long  NumofDummies;    +
  tFCEVector  Dummies[16];  +
  long  NumParts;      +
  tFCEVector  Parts[64];    +
  long  Vert1stPart[64];  +
  long  VertNumPart[64];  +
  long  Tria1stPart[64];  +
  long  TriaNumPart[64];  +
  long  NumColors;      +
  tColor  PrimColors[16];    +
  tColor  IntrColors[16];    +
  tColor  SecnColors[16];    +
  tColor  HairColors[16];    +
  char  UnknowSpace[260];  ---
  char  DummyNames[16*64];  +
  char  PartNames[64*64];  +
  char  UnknowSector[528];  ---
*/

  long type = 0;
  AfxSetResourceHandle(DllHIns);
  CFCETypeDialog diag(&type);
  diag.DoModal();
  AfxSetResourceHandle(AppHIns);


  int        i, j, Counter;
  int        VertCounter;
  CString      Name, Name2;
  BOOL      Found;
  long      TotalVertAmount = 0;
  long      TotalTriaAmount = 0;
  long      TotalObjects  = 0;
  nfs4fcehead    FileHead;
  int        ObjType;
  int        DamagedIndex  = 0;
  tPOINT      CenterPoint;
  tPOINT      Max, Min;
  FileHead.NumofDummies = 0;

  tUnrecognizedData *ColorInfo;
  
  long TempColorIndex = Z3D_UNDATA_COLORS;
  
  FileHead.NumColors = 0;

  if (pwnd->MessageBox("Use Default colors set (8 colors from Chevrolet) ?", "Colors source", MB_ICONQUESTION|MB_YESNO)
    == IDNO)
  {
    FileHead.NumColors = (long)(SYSTEMREQUESTPROC)RequestProc(
            Z3D_REQUEST_SELECTUNSUPPORTEDDATA,
            Objects,
            "Select source of a color information:",
            &TempColorIndex,
            NULL);
    if (FileHead.NumColors < 0) FileHead.NumColors = 0;
    else
      ColorInfo = &(UnData->Datas[FileHead.NumColors]);
  }
  else
  {
    FileHead.NumColors = -1; //indicator to delete ColorInfo
    ColorInfo = new tUnrecognizedData;
    ColorInfo->Allocate(16*16+1, 16*16+1);
    ColorInfo->SetName(CString("Need for Speed 4: Default colors set"));
    ColorInfo->SetID(Z3D_UNDATA_COLORS);

    typedef BYTE  ColSet[257];
    /////////////////////////////////////////////////////////////////////////
    // Default set of colors (from Corvette) (sorted to adapt UnData storage.
    ColSet set = 
    {
    0x08,0x00,0xFF,0x80,0x81,0xB5,0x06,0x4A,0x80,0x1B,0x7A,0x72,0x7F,0x1E,0xBE,0x46,
    0x7F,0x2D,0x3C,0x5A,0x7F,0xB5,0x06,0x4A,0x80,0x25,0x02,0x1A,0x7F,0x1E,0xBE,0x46,
    0x7F,0x99,0xDF,0x67,0x7F,0xB5,0x06,0x4A,0x80,0x25,0x02,0x1A,0x7F,0x1E,0xBE,0x46,
    0x7F,0x92,0x09,0x9C,0x7F,0xB5,0x06,0x4A,0x80,0x25,0x02,0x1A,0x7F,0x1E,0xBE,0x46,
    0x7F,0x7F,0x00,0x05,0x7F,0xB5,0x06,0x4A,0x80,0x1B,0x7A,0x72,0x7F,0x1E,0xBE,0x46,
    0x7F,0xAB,0xB8,0x2D,0x7F,0xB5,0x06,0x4A,0x80,0x25,0x02,0x1A,0x7F,0x1E,0xBE,0x46,
    0x7F,0xFC,0xFE,0x3D,0x81,0xB5,0x06,0x4A,0x80,0x1B,0x7A,0x72,0x7F,0x1E,0xBE,0x46,
    0x7F,0x87,0x1F,0x6B,0x7F,0xB5,0x06,0x4A,0x80,0x25,0x02,0x1A,0x7F,0x1E,0xBE,0x46,
    0x7F,0x7F,0x01,0x98,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,0x02,0x03,0x64,
    0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00};
  
    BYTE* pSet = new ColSet;
    memcpy(ColorInfo->GetDataPtr(), (char*)&set, 257);
  }

  if (ColorInfo)
  {
    BOOL needdelete = (FileHead.NumColors==-1);
    BYTE*  pBuf = (BYTE*)ColorInfo->GetDataPtr();
    FileHead.NumColors = (long)ColorInfo->GetBYTE(0);
    for (long count=0; count < 16; count++)
    {
      FileHead.PrimColors[count].Hue      = pBuf[1+count*16];
      FileHead.PrimColors[count].Saturation  = pBuf[1+count*16+1];
      FileHead.PrimColors[count].Brightness  = pBuf[1+count*16+2];
      FileHead.PrimColors[count].Transparency  = pBuf[1+count*16+3];
      FileHead.IntrColors[count].Hue      = pBuf[1+count*16+4];
      FileHead.IntrColors[count].Saturation  = pBuf[1+count*16+5];
      FileHead.IntrColors[count].Brightness  = pBuf[1+count*16+6];
      FileHead.IntrColors[count].Transparency  = pBuf[1+count*16+7];
      FileHead.SecnColors[count].Hue      = pBuf[1+count*16+8];
      FileHead.SecnColors[count].Saturation  = pBuf[1+count*16+9];
      FileHead.SecnColors[count].Brightness  = pBuf[1+count*16+10];
      FileHead.SecnColors[count].Transparency  = pBuf[1+count*16+11];
      FileHead.HairColors[count].Hue      = pBuf[1+count*16+12];
      FileHead.HairColors[count].Saturation  = pBuf[1+count*16+13];
      FileHead.HairColors[count].Brightness  = pBuf[1+count*16+14];
      FileHead.HairColors[count].Transparency  = pBuf[1+count*16+15];
    }
    if (needdelete) delete ColorInfo;
  }//do not discard color info
  //// reseting names string ////
  for (i = 0; i < 64; i++)
    for (j = 0; j < 64; j++)
      FileHead.PartNames[i][j] = (char)0;
  //// reseting dummies string ////
  for (i = 0; i < 16; i++)
    for (j = 0; j < 64; j++)
      FileHead.DummyNames[i][j] = (char)0;
  
  for (Counter = 0; Counter < Objects->ObjAmount; Counter++)
  {
    Name = Objects->ObjSet[Counter].ObjectName;
    if (Name.Find("Dummy:")==0)
    {//dummy object
      if (FileHead.NumofDummies < 16)
      {
        // Writing Dummie Names.....
        Name = Name.Right(Name.GetLength()-6);
        if (Name.Find('[',0) > 0)
          Name = Name.Left(Name.Find('[',0));
        strcpy(FileHead.DummyNames[FileHead.NumofDummies], Name.GetBuffer(16));

        CenterPoint = GetMatrixPosition(Objects->ObjSet[Counter].mTransform);
        FileHead.Dummies[FileHead.NumofDummies].X = -CenterPoint.x;
        FileHead.Dummies[FileHead.NumofDummies].Y = CenterPoint.y;
        FileHead.Dummies[FileHead.NumofDummies].Z = CenterPoint.z;
        FileHead.NumofDummies++;
      }
      else
        if (FileHead.NumofDummies == 16)
        {
          ShowFailMessage(pwnd, "not all Dummy: objects were saved", "too much of them: 16 is maximum.", MB_ICONEXCLAMATION);
          FileHead.NumofDummies = 17;//!!!!!
        }
    }//Dummy: object
    else
    if (Name.Find("[D]")==-1)
    {//Non [D]amaged object
      FileHead.Vert1stPart[TotalObjects] = TotalVertAmount;
      FileHead.VertNumPart[TotalObjects] = Objects->ObjSet[Counter].VertTable->VertAmount;
      FileHead.Tria1stPart[TotalObjects] = TotalTriaAmount;
      FileHead.TriaNumPart[TotalObjects] = Objects->ObjSet[Counter].FaceTable->FaceAmount;

      TotalVertAmount+=Objects->ObjSet[Counter].VertTable->VertAmount;
      if (Objects->ObjSet[Counter].FaceTable->FaceAmount==0)
        AfxMessageBox("Kidding??? Strange object found=)");
      TotalTriaAmount+=Objects->ObjSet[Counter].FaceTable->FaceAmount;
      TotalObjects++;
    }//Non [D]amaged object
  }//objects counter
  TotalObjects--;
  
  FileHead.NumofVert = TotalVertAmount;
  FileHead.NumofTria = TotalTriaAmount;
  tFCEVector  *Vectors = new tFCEVector[FileHead.NumofVert];
  tFCEVector  *Normals = new tFCEVector[FileHead.NumofVert];
  tFCEVector  *DamVectors = new tFCEVector[FileHead.NumofVert];
  tFCEVector  *DamNormals = new tFCEVector[FileHead.NumofVert];
  DWORD  *RotMap = new DWORD[FileHead.NumofVert];
  tFCETrian  *Trian = new tFCETrian[FileHead.NumofTria];

  long  TotalTextures = 0;
  long  TexturePages[255];
  CString name1, name2;

  for (Counter = 0; Counter < 255; Counter++)
    TexturePages[Counter] = -1;

  // this sets will contain all vert/tria information for the file //

  //Now filling this sets with vert/tria information...
  TotalObjects      = 0;
  for (Counter = 0; Counter < Objects->ObjAmount; Counter++)
  {
    Name = Objects->ObjSet[Counter].ObjectName;
    ObjType = Name.Find("[D]");
    if (Name.Find("Dummy:")!=0)
    {///// non light /////
      if (ObjType==-1)
      {//Non [D]amaged object
        // Writing Parts Names.....
        if (Name.Find('[') > 0)
          Name = Name.Left(Name.Find('['));
        strcpy(FileHead.PartNames[TotalObjects], Name.GetBuffer(64));

        
        ////Looking for corresponding damaged object...
        Name+="[D]";
        Found = FALSE;
        for (DamagedIndex = 0; DamagedIndex < Objects->ObjAmount && !Found; DamagedIndex++)
        {
          Name2 = Objects->ObjSet[DamagedIndex].ObjectName;
          Found = (Name2.Find(Name)>=0);
          // Looking for damaged Object...//
        }
        DamagedIndex--;
        if (DamagedIndex>=Objects->ObjAmount)
        {
        // this mean that we didn't found corresponding
        // damaged object, than damaged is the same (no
        // damage will happen in NFS4)... OK..    then=>
          DamagedIndex = Counter;
        }
        else
          if ((Objects->ObjSet[DamagedIndex].VertTable->VertAmount!=Objects->ObjSet[Counter].VertTable->VertAmount) ||
            (Objects->ObjSet[DamagedIndex].FaceTable->FaceAmount!=Objects->ObjSet[Counter].FaceTable->FaceAmount))
          // this mean that normal and damaged objects have
          // different amount of vertices and/or triangles
          {
            DamagedIndex = Counter;
          }
        CenterPoint = GetMatrixPosition(Objects->ObjSet[Counter].mTransform);
        /*
        if (Name.Find(":OD") == 0)
        {//recalculating center for proper rotation of vertices;
          int vertamount=0;
          tPOINT max, min;
          for (VertCounter = 0; VertCounter < Objects->ObjSet[Counter].VertTable->VertAmount; VertCounter++)
            if (Objects->ObjSet[Counter].VertTable->Table[VertCounter].HasFlag(Z3D_FLAG_MISC1))
            {
              if (vertamount == 0)
              {
                max.x = Objects->ObjSet[Counter].VertTable->Table[VertCounter].X;
                max.y = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Y;
                max.z = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Z;
                min = max;
                vertamount=1;
              }
              else
              {
                if (max.x < Objects->ObjSet[Counter].VertTable->Table[VertCounter].X)
                  max.x = Objects->ObjSet[Counter].VertTable->Table[VertCounter].X;
                if (max.y < Objects->ObjSet[Counter].VertTable->Table[VertCounter].Y)
                  max.y = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Y;
                if (max.z < Objects->ObjSet[Counter].VertTable->Table[VertCounter].Z)
                  max.z = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Z;
                if (min.x > Objects->ObjSet[Counter].VertTable->Table[VertCounter].X)
                  min.x = Objects->ObjSet[Counter].VertTable->Table[VertCounter].X;
                if (min.y > Objects->ObjSet[Counter].VertTable->Table[VertCounter].Y)
                  min.y = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Y;
                if (min.z > Objects->ObjSet[Counter].VertTable->Table[VertCounter].Z)
                  min.z = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Z;
              }
            }
          CenterPoint.x = (max.x+min.x)*0.5f;
          CenterPoint.y = (max.y+min.y)*0.5f;
          CenterPoint.z = (max.z+min.z)*0.5f;
        }//recalculating center for proper rotation of vertices;
        */

        FileHead.Parts[TotalObjects].X = -CenterPoint.x;
        FileHead.Parts[TotalObjects].Y = CenterPoint.y;
        FileHead.Parts[TotalObjects].Z = CenterPoint.z;
        for (VertCounter = 0; VertCounter < Objects->ObjSet[Counter].VertTable->VertAmount; VertCounter++)
        {//Recording vert/normals
          Vectors[FileHead.Vert1stPart[TotalObjects]+VertCounter].X = -(Objects->ObjSet[Counter].VertTable->Table[VertCounter].X-CenterPoint.x);
          Vectors[FileHead.Vert1stPart[TotalObjects]+VertCounter].Y = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Y-CenterPoint.y;
          Vectors[FileHead.Vert1stPart[TotalObjects]+VertCounter].Z = Objects->ObjSet[Counter].VertTable->Table[VertCounter].Z-CenterPoint.z;
          RotMap[FileHead.Vert1stPart[TotalObjects]+VertCounter] = Objects->ObjSet[Counter].VertTable->Table[VertCounter].HasFlag(Z3D_FLAG_MISC1) ? 0x00000000 : 0x00000004;
          Normals[FileHead.Vert1stPart[TotalObjects]+VertCounter].X = -(Objects->ObjSet[Counter].VertTable->Table[VertCounter].NormalX);
          Normals[FileHead.Vert1stPart[TotalObjects]+VertCounter].Y = Objects->ObjSet[Counter].VertTable->Table[VertCounter].NormalY;
          Normals[FileHead.Vert1stPart[TotalObjects]+VertCounter].Z = Objects->ObjSet[Counter].VertTable->Table[VertCounter].NormalZ;
          DamVectors[FileHead.Vert1stPart[TotalObjects]+VertCounter].X = -(Objects->ObjSet[DamagedIndex].VertTable->Table[VertCounter].X-CenterPoint.x);
          DamVectors[FileHead.Vert1stPart[TotalObjects]+VertCounter].Y = Objects->ObjSet[DamagedIndex].VertTable->Table[VertCounter].Y-CenterPoint.y;
          DamVectors[FileHead.Vert1stPart[TotalObjects]+VertCounter].Z = Objects->ObjSet[DamagedIndex].VertTable->Table[VertCounter].Z-CenterPoint.z;
          DamNormals[FileHead.Vert1stPart[TotalObjects]+VertCounter].X = -(Objects->ObjSet[DamagedIndex].VertTable->Table[VertCounter].NormalX);
          DamNormals[FileHead.Vert1stPart[TotalObjects]+VertCounter].Y = Objects->ObjSet[DamagedIndex].VertTable->Table[VertCounter].NormalY;
          DamNormals[FileHead.Vert1stPart[TotalObjects]+VertCounter].Z = Objects->ObjSet[DamagedIndex].VertTable->Table[VertCounter].NormalZ;
        }//Recording vert/normals
        for (VertCounter = 0; VertCounter < Objects->ObjSet[Counter].FaceTable->FaceAmount; VertCounter++)
        {//Recording TRIANGLES!!!!
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].I1 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].I1;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].I3 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].I2;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].I2 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].I3;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].V1 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].V1;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].V3 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].V2;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].V2 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].V3;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].U1 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].U1;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].U3 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].U2;
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].U2 = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].U3;
          if (type==1)
          {
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[0] = (char)0; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[1] = (char)255; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[2] = (char)0; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[3] = (char)255; 
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[4] = (char)0; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[5] = (char)255; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[6] = (char)0; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[7] = (char)255; 
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[8] = (char)0; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[9] = (char)255; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[10] = (char)0; Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[11] = (char)255; 
          }
          else//MCO
            memset(&Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Unknow[0], 0xFF, 12);
// I use:
//  0x00 - chrome
//  0x01 - nochrome
//  0x02 - highchrome
//  0x04 - no cull
//  0x08 - highchrome
          Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Smoothing = Objects->ObjSet[Counter].FaceTable->Table[VertCounter].GetMiscFlag(0) & ~0x0100000F;
          if ((Objects->ObjSet[Counter].FaceTable->Table[VertCounter].nRenderFlags & NoBlend) == NoBlend)
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Smoothing |= 0x01;
          if ((Objects->ObjSet[Counter].FaceTable->Table[VertCounter].nRenderFlags & HighBlend) == HighBlend)
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Smoothing |= 0x02;
          if ((Objects->ObjSet[Counter].FaceTable->Table[VertCounter].nRenderFlags & CullNone) == CullNone)
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Smoothing |= 0x04;
          if ((Objects->ObjSet[Counter].FaceTable->Table[VertCounter].nRenderFlags & SemiTransparent) == SemiTransparent)
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].Smoothing |= 0x08;


          if (Objects->ObjSet[Counter].FaceTable->Table[VertCounter].Material>=0)
          {
            if (d3d->Materials.Materials[Objects->ObjSet[Counter].FaceTable->Table[VertCounter].Material].MatParams.PrimTexture>=0)
            {//textured
              if (TexturePages[d3d->Materials.Materials[Objects->ObjSet[Counter].FaceTable->Table[VertCounter].Material].MatParams.PrimTexture] == -1)
              {//first time this texture match
                TexturePages[d3d->Materials.Materials[Objects->ObjSet[Counter].FaceTable->Table[VertCounter].Material].MatParams.PrimTexture] = TotalTextures;
                TotalTextures++;
              }//first time this texture match
              Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].TexturePage = (long)TexturePages[d3d->Materials.Materials[Objects->ObjSet[Counter].FaceTable->Table[VertCounter].Material].MatParams.PrimTexture];
            }//textured
            else Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].TexturePage = 0L;
          }//not default material
          else
            Trian[FileHead.Tria1stPart[TotalObjects]+VertCounter].TexturePage = 0L;
        }//Recording TRIANGLES

        TotalObjects++;
      }//Non [D]amaged object
    }///// non light /////
    
  }//objects counter


  Objects->GetMaxMinPoints(&Max, &Min);
  FileHead.RealXHalfSize  = fdivide(Max.x-Min.x, 2.0f);
  FileHead.RealYHalfSize  = fdivide(Max.y-Min.y, 2.0f);
  FileHead.RealZHalfSize  = fdivide(Max.z-Min.z, 2.0f);
  
  FileHead.NumParts  = TotalObjects;
  if (FileHead.NumofDummies > 16)
    FileHead.NumofDummies = 16;

  FileHead.HeaderID  = (type==1) ? NFS4HEADERID1 : MCO_HEADERID1;
  FileHead.HeaderID2  = NFS4HEADERID2;
  FileHead.NumofArt  = 1;//single TexturePage ????!!!!!!!!!!!
  FileHead.VertOffset  = 0;
  FileHead.NormalsOffset  = FileHead.NumofVert* sizeof(tFCEVector);
  FileHead.TriaOffset  =  FileHead.NormalsOffset + FileHead.NumofVert* sizeof(tFCEVector);
  
  FileHead.TempStorage1offset    = FileHead.TriaOffset + FileHead.NumofTria* sizeof(tFCETrian);
  FileHead.TempStorage2offset    = FileHead.TempStorage1offset + FileHead.NumofVert * 32;
  FileHead.TempStorage3offset    = FileHead.TempStorage2offset + FileHead.NumofVert * 12;
  FileHead.UndamagedVertOffset  = FileHead.TempStorage3offset + FileHead.NumofVert * 12;
  FileHead.UndamagedNormOffset  = FileHead.UndamagedVertOffset + FileHead.NumofVert * 12;
  FileHead.DamagedVertOffset    = FileHead.UndamagedNormOffset + FileHead.NumofVert * 12;
  FileHead.DamagedNormOffset    = FileHead.DamagedVertOffset + FileHead.NumofVert * 12;
  FileHead.UnknowAreaOffset    = FileHead.DamagedNormOffset + FileHead.NumofVert * 12;
  FileHead.MovementsOffset    = FileHead.UnknowAreaOffset + FileHead.NumofVert * 4;//unknow area is not tFCEVectors....
  FileHead.Unknow1Offset      = FileHead.MovementsOffset + FileHead.NumofVert * 4;
  FileHead.Unknow2Offset      = FileHead.Unknow1Offset + FileHead.NumofVert * 4;

  
  ///////////////////////////////////////
  //////    Lets Write IT!!!  ///////
  ///////////////////////////////////////
  OutFile.Write(&FileHead, sizeof(nfs4fcehead));


  OutFile.WriteHuge(Vectors,  sizeof(tFCEVector)*FileHead.NumofVert);
  OutFile.WriteHuge(Normals,  sizeof(tFCEVector)*FileHead.NumofVert);
  OutFile.WriteHuge(Trian,  sizeof(tFCETrian)*FileHead.NumofTria);
  /// instead ///
  OutFile.WriteHuge(Vectors,  sizeof(tFCEVector)*FileHead.NumofVert);//12
  OutFile.WriteHuge(Vectors,  sizeof(tFCEVector)*FileHead.NumofVert);//24
  OutFile.WriteHuge(Vectors,  FileHead.NumofVert*8);//32

  OutFile.WriteHuge(Vectors,  sizeof(tFCEVector)*FileHead.NumofVert);//12
  OutFile.WriteHuge(Vectors,  sizeof(tFCEVector)*FileHead.NumofVert);//12
  //undamaged
  OutFile.WriteHuge(Vectors,  sizeof(tFCEVector)*FileHead.NumofVert);
  OutFile.WriteHuge(Normals,  sizeof(tFCEVector)*FileHead.NumofVert);
  OutFile.WriteHuge(DamVectors,  sizeof(tFCEVector)*FileHead.NumofVert);
  OutFile.WriteHuge(DamNormals,  sizeof(tFCEVector)*FileHead.NumofVert);
  
  //UnknowAreaOffset
  OutFile.WriteHuge(Vectors,  FileHead.NumofVert*4);//4
  
  //Rotateability
  OutFile.WriteHuge(RotMap, FileHead.NumofVert*4);

  //Unknow1
  OutFile.WriteHuge(Vectors,  FileHead.NumofVert*4);//4
  //Unknow2
  OutFile.WriteHuge(Vectors,  FileHead.NumofVert*12);//12


  if (Vectors)  delete[] Vectors;
  if (Normals)  delete[] Normals;
  if (DamVectors)  delete[] DamVectors;
  if (DamNormals)  delete[] DamNormals;
  if (Trian)    delete[] Trian;
  if (RotMap)    delete[] RotMap;

  OutFile.Close();
  return 1;
}//Export




//////////////////////////////////////////////////////////////
//  GetMenuFolder writes a name of subfolder. It can be your
//  own folder. or it can be emty. There are some standart
//  folders for MenuBAR:
char* CALLBACK GetMenuFolder(DWORD num)
{
  //////////////////////////////
  // valid MainMenu folders are:
  // &File
  // &View
  // &Tools
  // &Options
  // &Plugins
  // &Help
  if (num==0)
    return "&Plugins\0";
  return NULL;
}


/////////////////////////////////////////////////////////////////
//  void OnLoad processes creation of the corresponding floatbar.
//  and attaching of it into the main window;
DWORD CALLBACK OnLoadFloatBar(
            DWORD        num,
            CRollUpBar**    DB,
            CWnd        *pwnd,
            HINSTANCE      AppHandle,
            HINSTANCE      DLLHandle)
{
  if (num != 0)
    return 0;

  z3dProcsSet*  pProcs = NULL;
  CRollUpBar*    pDB;
  if (DLLHandle && pwnd)
  {
    AfxSetResourceHandle(DLLHandle);// Tell system to use DLL's resources.
    pDB = new CFCEFloatBar(pwnd);
    if (pDB)
    {
      pDB->EnableDocking(0);
      pDB->ShowWindow(SW_HIDE);
    }
    /////////////// initialisate global variables;
    dll = DLLHandle;
    app = AppHandle;

    AfxSetResourceHandle(AppHandle);// Set it back before leaving.
  }
  else
  {
    pwnd->MessageBox("Resources not found or Editor window was not created!","Internal Error:", MB_OK | MB_ICONEXCLAMATION);
    pDB = NULL;
    return 0;
  }

  if (pDB) *DB = pDB;
  return 1; //ok, dockable;
}



extern DWORD CALLBACK FCECustomSettingsOnSceneChanged(tProcParams* params);
/////////////////////////////////////////////////////////////////
//  Activate processes initialization of floatbar/adjusting global
//  variables and other stuff.
DWORD CALLBACK FCECustomSettingsActivate(tProcParams* params)
{
  return FCECustomSettingsOnSceneChanged(params);
}


/////////////////////////////////////////////////////////////////
//  Deactivates floatbar
DWORD CALLBACK FCECustomSettingsDeactivate(tProcParams* params)
{
  return 0;
}
